<?php
if (!defined('ABSPATH')) {
    exit;
}

class WC_Xendit_Invoice extends WC_Payment_Gateway
{
    const DEFAULT_MAXIMUM_AMOUNT = 1000000000;
    const DEFAULT_MINIMUM_AMOUNT = 1;
    const DEFAULT_EXTERNAL_ID_VALUE = 'woocommerce-xendit';
    const DEFAULT_CHECKOUT_FLOW = 'CHECKOUT_PAGE';

    const API_KEY_FIELDS = array('dummy_api_key', 'dummy_secret_key', 'dummy_api_key_dev', 'dummy_secret_key_dev');

    /**
     * @var WC_Xendit_Invoice
     */
    private static $_instance;

    /** @var bool $isActionCalled */
    public $isActionCalled = false;

    /** @var string $method_code */
    public $method_code;

    /** @var string $developmentmode */
    public $developmentmode = '';

    /** @var string $showlogo */
    public $showlogo = 'yes';

    /** @var string $success_response_xendit */
    public $success_response_xendit = 'COMPLETED';

    /** @var string $success_payment_xendit */
    public $success_payment_xendit;

    /** @var string $responce_url_sucess */
    public $responce_url_sucess;

    /** @var string $checkout_msg */
    public $checkout_msg = 'Thank you for your order, please follow the account numbers provided to pay with secured Xendit.';

    /** @var string $xendit_callback_url */
    public $xendit_callback_url;

    /** @var string $generic_error_message */
    public $generic_error_message = 'We encountered an issue while processing the checkout. Please contact us. ';

    /** @var string $xendit_status */
    public $xendit_status;

    /** @var array $msg */
    public $msg = ['message' => '', 'class' => ''];

    /** @var string $external_id_format */
    public $external_id_format;

    /** @var string $redirect_after */
    public $redirect_after;

    /** @var string $for_user_id */
    public $for_user_id;

    /** @var string $enable_xenplatform */
    public $enable_xenplatform;

    /** @var string $publishable_key */
    public $publishable_key;

    /** @var string $secret_key */
    public $secret_key;

    /** @var WC_Xendit_PG_API $xenditClass */
    public $xenditClass;

    /** @var false|mixed|null $oauth_data */
    public $oauth_data;

    /** @var string $oauth_link */
    public $oauth_link;

    /** @var bool $is_connected */
    public $is_connected = false;

    /** @var array|mixed $merchant_info */
    public $merchant_info;

    /** @var int $setting_processed */
    public static $setting_processed = 0;

    /**
     * @var string $method_type
     */
    public $method_type = '';

    /** @var string $default_title */
    public $default_title = '';

    /**
     * @var int $DEFAULT_MAXIMUM_AMOUNT
     */
    public $DEFAULT_MAXIMUM_AMOUNT = 0;

    /**
     * @var int $DEFAULT_MINIMUM_AMOUNT
     */
    public $DEFAULT_MINIMUM_AMOUNT = 0;

    /**
     * Constructor
     */
    public function __construct()
    {
        global $woocommerce;

        $this->id = 'xendit_gateway';
        $this->has_fields = true;
        $this->method_title = 'Xendit';
        $this->default_title = 'Xendit Payment Gateway';
        $this->method_type = $this->method_title;
        /* translators: %1$s: Payment Method Accepted, %2%s: Xendit Login Link, %3%s: Xendit Register Link, %4%s: Developer Link. */
        $this->method_description = sprintf(wp_kses(__('Collect payment from %1$s on checkout page and get the report realtime on your Xendit Dashboard. <a href="%2$s" target="_blank">Sign In</a> or <a href="%3$s" target="_blank">sign up</a> on Xendit and integrate with your <a href="%4$s" target="_blank">Xendit keys</a>', 'woo-xendit-virtual-accounts'), ['a' => ['href' => true, 'target' => true]]), 'Bank Transfer (Virtual Account), Credit Card, Direct Debit, EWallet, QR Code & PayLater', 'https://dashboard.xendit.co/auth/login', 'https://dashboard.xendit.co/register', 'https://dashboard.xendit.co/settings/developers#api-keys');
        $this->method_code = strtoupper($this->method_title);
        $this->enabled = $this->get_option('enabled');

        $this->supports = array(
            'products'
        );

        $this->init_form_fields();
        $this->init_settings();

        // user setting variables
        $this->title = $this->get_xendit_title();
        $this->description = $this->get_xendit_description();

        $this->DEFAULT_MAXIMUM_AMOUNT = self::DEFAULT_MAXIMUM_AMOUNT;
        $this->DEFAULT_MINIMUM_AMOUNT = self::DEFAULT_MINIMUM_AMOUNT;

        $this->developmentmode = $this->get_option('developmentmode');

        $this->success_payment_xendit = $this->get_option('success_payment_xendit');
        $this->responce_url_sucess = $this->get_option('responce_url_calback');
        $this->xendit_callback_url = home_url() . '/?wc-api=wc_xendit_callback&xendit_mode=xendit_invoice_callback';

        $this->xendit_status = $this->developmentmode == 'yes' ? "[Development]" : "[Production]";

        $this->external_id_format = !empty($this->get_option('external_id_format')) ? $this->get_option('external_id_format') : self::DEFAULT_EXTERNAL_ID_VALUE;
        $this->redirect_after = !empty($this->get_option('redirect_after')) ? $this->get_option('redirect_after') : self::DEFAULT_CHECKOUT_FLOW;
        $this->for_user_id = $this->get_option('on_behalf_of');
        $this->enable_xenplatform = $this->for_user_id ? 'yes' : $this->get_option('enable_xenplatform');

        // API Key
        $this->publishable_key = $this->developmentmode == 'yes' ? $this->get_option('api_key_dev') : $this->get_option('api_key');
        $this->secret_key = $this->developmentmode == 'yes' ? $this->get_option('secret_key_dev') : $this->get_option('secret_key');

        $this->xenditClass = new WC_Xendit_PG_API();
        $this->oauth_data = WC_Xendit_Oauth::getXenditOAuth();

        // Generate Validation Key
        if (empty(WC_Xendit_Oauth::getValidationKey())) {
            $key = md5(wp_rand());
            WC_Xendit_Oauth::updateValidationKey($key);
        }

        // Generate OAuth link
        $this->oauth_link = "https://dashboard.xendit.co/oauth/authorize";
        $this->oauth_link .= "?client_id=906468d0-fefd-4179-ba4e-407ef194ab85"; // initiate with prod client
        $this->oauth_link .= "&response_type=code&state=WOOCOMMERCE|"
            . WC_Xendit_Oauth::getValidationKey() . "|"
            . home_url() . "?wc-api=wc_xendit_oauth|".WC_XENDIT_PG_VERSION;
        $this->oauth_link .= "&redirect_uri=https://tpi-gateway.xendit.co/tpi/authorization/xendit/redirect/v2";

        add_action('woocommerce_update_options_payment_gateways_' . $this->id, array($this, 'process_admin_options'));
        add_action('woocommerce_receipt_' . $this->id, array(&$this, 'receipt_page'));

        add_filter('woocommerce_available_payment_gateways', array(&$this, 'check_gateway_status'));
        add_action('woocommerce_order_status_changed', array(&$this, 'expire_invoice_when_order_cancelled'), 10, 3);
        wp_register_script('sweetalert', plugins_url('assets/js/frontend/sweetalert.min.js', WC_XENDIT_PG_MAIN_FILE), null, WC_XENDIT_PG_VERSION, true);
        wp_enqueue_script('sweetalert');

        // Init payment channels
        $this->init_activate_payment_channel();
    }

    /**
     * @return WC_Xendit_Invoice
     */
    public static function instance()
    {
        if (is_null(self::$_instance)) {
            self::$_instance = new self();
        }
        return self::$_instance;
    }

    /**
     * @return bool
     */
    protected function onboarded_payment_channel(): bool
    {
        if (get_option('xendit_onboarding_payment_channel') == 1) {
            return true;
        }
        $this->update_option('enabled', 'yes'); // Enable Xendit_Gateway

        return update_option('xendit_onboarding_payment_channel', 1);
    }
    

    /**
     * Used to enable the default activate channel when onboarding
     *
     * @return void
     */
    protected function init_activate_payment_channel()
    {
        if (!is_admin()) {
            return;
        }

        $this->onboarded_payment_channel();
    }

    /**
     * @return string
     */
    protected function generate_api_key_settings_html(): string
    {
        $form_fields = $this->form_fields;

        foreach ($form_fields as $index => $field) {
            if (!in_array($index, array_merge(self::API_KEY_FIELDS, ['developmentmode', 'api_keys_help_text']))) {
                unset($form_fields[ $index ]);
            }
        }

        return $this->generate_settings_html($form_fields, false);
    }

    /**
     * @return void
     */
    protected function get_xendit_connection()
    {
        try {
            if (empty(get_transient('xendit_merchant_info'))) {
                $response = $this->xenditClass->getMerchant();
                if (!empty($response['error_code'])) {
                    throw new Exception($response['message']);
                }

                if (!empty($response['business_id'])) {
                    $this->merchant_info = $response;
                    set_transient('xendit_merchant_info', $response, 3600);
                    $this->is_connected = true;
                }
            } else {
                $this->merchant_info = get_transient('xendit_merchant_info');
                $this->is_connected = !empty($this->merchant_info['business_id']);
            }
        } catch (\Exception $e) {
            WC_Admin_Settings::add_error(esc_html($e->getMessage()));
        }
    }

    /**
     * @return void
     */
    protected function initialize_xendit_onboarding_info()
    {
        echo wp_kses("<h2>Xendit</h2><p style='margin-bottom: 10px;'>".
            __('Accept payments with Xendit. See our 
                <a href="https://docs.xendit.co/integrations/woocommerce/steps-to-integrate" target="_blank">documentation</a> for the full guide', 
                'woo-xendit-virtual-accounts')."</p><br />", 
                ['h2' => true, 'p' => true, 'a' => ['href' => true, 'target' => true]]
            );

        if (!$this->is_connected) {
            $top = '<div class="oauth-container">';
            $top .= "<button class='components-button is-primary' id='woocommerce_xendit_connect_button'>" . esc_html(__('Connect to Xendit', 'woo-xendit-virtual-accounts')) . "</button>";
            $top .= '
            <ul>
                    <li>'. esc_html('1. Click "Connect to Xendit"') .'</li>
                    <li>'. esc_html('2. Log in to your Xendit dashboard (If you haven\'t)') .'</li>
                    <li>'. esc_html('3. Click "Allow"') .'</li>
                    <li>'. esc_html('4. Done') .'</li>
            </ul>';
            $top .= '<em>If you\'re having trouble with the "Connect" button, click <a href="#" id="woocommerce_xendit_connect_api_key_button">here</a> to connect manually using your API keys.</em>';
            $top .= '</div>';

            $top_allowed_html = [
                'div' => [
                    'class' => true,
                    'style' => true
                ],
                'button' => [
                    'class' => true,
                    'id' => true
                ],
                'ul' => true,
                'li' => true,
                'em' => true,
                'a' => [
                    'href' => true,
                    'id' => true
                ]
            ];            

            $content = '<div class="api-keys-container" style="display: none;"><table class="form-table">';
            $content .= '<a href="#" id="woocommerce_xendit_connect_oauth_button"><< '. esc_html('Back') .'</a>';
            $content .= $this->generate_api_key_settings_html();
            $content .= '</table>';

            $content .= '</div>';

            $content .= '
                    <style>
                            .submit{display:none;}
                    </style>
                            <script>
                            jQuery(document).ready(function($) {
                                $("#woocommerce_xendit_connect_api_key_button").on("click", function() {
                                    $(".oauth-container").hide();
                                    $(".api-keys-container").show();
                                    $(".submit").show();
                                });
                                $("#woocommerce_xendit_connect_oauth_button").on("click", function() {
                                    $(".oauth-container").show();
                                    $(".api-keys-container").hide();
                                    $(".submit").hide();
                                });
                            });
                    </script>
                ';
        
            $connect_allowed_html =[
                'div' => [
                    'class' => true,
                    'style' => true
                ],
                'a' => [
                    'href' => true,
                    'id' => true
                ],
                'table' => [
                    'class' => true
                ],
                'style' => true,
                'script' => true,
                'tr' => [
                    'valign' => true
                ],
                'th' => [
                    'scope' => true,
                    'class' => true
                ],
                'td' => [
                    'class' => true
                ],
                'fieldset' => true,
                'legend' => [
                    'class' => true
                ],
                'span' => true,
                'input' => [
                    'class' => true,
                    'type' => true,
                    'name' => true,
                    'id' => true,
                    'value' => true,
                    'checked' => true,
                    'disabled' => true,
                    'style' => true,
                    'placeholder' => true
                ]

            ];

            $full_content = $top.$content;
            $allowed_html = array_merge($top_allowed_html, $connect_allowed_html);
            echo wp_kses($full_content, $allowed_html);
        } else {
            $content = "<button class='components-button is-secondary' disabled>" . esc_html__('Connected', 'woo-xendit-virtual-accounts') . "</button>";
            $content .= "<button class='components-button is-secondary' id='woocommerce_xendit_gateway_disconect_button'>" . esc_html__('Disconnect', 'woo-xendit-virtual-accounts') . "</button>";

            $disconnect_allowed_html =[
                'button' => [
                    'class' => true,
                    'disabled' => true,
                    'id' => true
                ]
            ];

            echo wp_kses($content,  $disconnect_allowed_html);
        }
    }

    /**
     * @return void
     */
    protected function show_merchant_info()
    {
        if (empty($this->is_connected)) {
            return;
        }
        ?>
            <h3 class="wc-settings-sub-title"><?php echo esc_html('Xendit Merchant Info')?></h3>
            <table class="form-table">
                <tbody>
                    <tr>
                        <th class="titledesc"><?php echo esc_html('Merchant')?></th>
                        <td>
                            <table>
                                <tr>
                                    <th class="titledesc" style="padding: 0;width: 85px;"><?php echo esc_html('Business ID')?></th>
                                    <td style="padding: 0;"><?php echo esc_html($this->merchant_info['business_id'] ?? '')?></td>
                                </tr>
                                <tr>
                                    <th class="titledesc" style="padding: 0;width: 85px;"><?php echo esc_html('Name')?></th>
                                    <td style="padding: 0;"><?php echo esc_html($this->merchant_info['name'] ?? '')?></td>
                                </tr>
                            </table>
                        </td>
                    </tr>
            </table>
            <?php
    }

    /**
     * @return void
     * @throws Exception
     */
    public function admin_options()
    {
        $this->get_xendit_connection(); // Always check the Xendit connection on the top of admin_options
        $this->initialize_xendit_onboarding_info();
        ?>

        <?php if ($this->is_connected) : ?>
        <table class="form-table">
            <?php $this->show_merchant_info(); ?>

            <?php
            // Remove secret key settings if merchant connected via OAuth
            if (empty($this->get_option('secret_key')) && empty($this->get_option('secret_key_dev'))) {
                unset($this->form_fields['dummy_secret_key']);
                unset($this->form_fields['dummy_secret_key_dev']);
            }
            ?>

            <?php $this->generate_settings_html(); ?>
        </table>
        <?php endif ?>

        <style>
            .xendit-ttl-wrapper {
                width: 400px;
                position: relative;
            }

            .xendit-ttl,
            .xendit-ext-id {
                width: 320px !important;
            }

            .xendit-form-suffix {
                width: 70px;
                position: absolute;
                bottom: 6px;
                right: 0;
            }
        </style>

        <script>
            jQuery(document).ready(function ($) {
                // always hide oauth fields
                $(".xendit-oauth").parents("tr").hide();

                <?php if (!$this->is_connected) : ?>
                    $(".api-keys-container").hide();
                <?php endif ?>

                // Disconect action
                let disconect_button = $('#woocommerce_xendit_gateway_disconect_button');
                disconect_button.on('click', function (e) {
                    e.preventDefault();
                    new swal({
                        title: "Are you sure you want to disconnect Xendit payment?",
                        text: "Transactions can no longer be made, and all settings will be lost.",
                        icon: "warning",
                        dangerMode: true,
                        buttons: ["Cancel", "Disconnect"],
                    })
                        .then((willDelete) => {
                            if (willDelete) {
                                disconect_button.text('Loading, please wait a moment...').attr('disabled', true);

                                fetch("<?php esc_html(home_url()); ?>/wp-json/xendit-wc/v1/disconnect", {
                                    method: "DELETE",
                                    headers: {
                                        "Content-Type": "application/json",
                                        "X-WP-Nonce": '<?php echo esc_html(wp_create_nonce('wp_rest'))?>'
                                    }
                                })
                                    .then((response) => response.json())
                                    .then(json => {
                                        switch (json.message) {
                                            case 'success':
                                                location.reload();
                                                break;
                                            case 'Sorry, you are not allowed to do that.':
                                                new swal({
                                                    type: 'error',
                                                    title: 'Failed',
                                                    text: 'Only Administrators and Shop Managers can disconnect'
                                                }).then(
                                                    function () {
                                                        location.reload();
                                                    }
                                                )
                                                break;
                                            default:
                                                new swal({
                                                    type: 'error',
                                                    title: 'Failed',
                                                    text: json.message
                                                }).then(
                                                    function () {
                                                        location.reload();
                                                    }
                                                )
                                                break;
                                        }
                                    })
                                    .catch(error => {
                                        new swal({
                                            type: 'error',
                                            title: 'Failed',
                                            text: 'Oops, something wrong happened! Please try again.'
                                        }).then(
                                            function () {
                                                location.reload();
                                            }
                                        )
                                    });
                            }
                        });
                });

                // Change send data value
                let send_data_button = $('#woocommerce_xendit_gateway_send_site_data_button');
                send_data_button.val('<?php echo esc_html(__('Send site data to Xendit', 'woo-xendit-virtual-accounts')); ?>');

                send_data_button.on('click', function (e) {
                    <?php
                    try {
                        $site_data = WC_Xendit_Site_Data::retrieve();
                        $this->xenditClass->createPluginInfo($site_data);
                        ?>
                            new swal({
                                type: 'success',
                                title: '<?php echo esc_html(__('Success', 'woo-xendit-virtual-accounts')); ?>',
                                text: '<?php echo esc_html(__('Thank you! We have successfully collected all the basic information that we need to assist you with any issues you may have. All data will remain private & confidential', 'woo-xendit-virtual-accounts')); ?>'
                            }).then(
                                function () {
                                    location.reload();
                                }
                            )
                        <?php
                    } catch (\Throwable $th) {
                        ?>
                            new swal({
                                type: 'error',
                                title: '<?php echo esc_html(__('Failed', 'woo-xendit-virtual-accounts')); ?>',
                                text: '<?php echo esc_html(__('Oops, something wrong happened! Please try again', 'woo-xendit-virtual-accounts')); ?>'
                            }).then(
                                function () {
                                    location.reload();
                                }
                            )
                        <?php
                    }
                    ?>
                });

                let xendit_connect_button = $('#woocommerce_xendit_connect_button');
                xendit_connect_button.on('click', function (e) {
                    e.preventDefault();
                    window.open("<?php echo esc_url_raw(sanitize_url($this->oauth_link)); ?>", '_blank').focus();

                    new swal({
                        title: "<?php echo esc_html__('Loading', 'woo-xendit-virtual-accounts'); ?> ...",
                        text: "<?php echo esc_html__('Please finish your integration on Xendit', 'woo-xendit-virtual-accounts'); ?>",
                        buttons: ["Cancel", false],
                        closeOnClickOutside: false,
                    }).then(
                        function () {
                            location.reload();
                        }
                    );

                    // Check OAuth status every 5 seconds
                    let checkOauthStatusInterval = setInterval(() => {
                        fetch("<?php echo esc_html(home_url()); ?>/wp-json/xendit-wc/v1/oauth_status", {
                            method: "GET",
                            headers: {
                                "Content-Type": "application/json",
                                "X-WP-Nonce": '<?php echo esc_html(wp_create_nonce('wp_rest'))?>'
                            }
                        })
                            .then((response) => response.json())
                            .then(json => {
                                if (json.is_connected) {
                                    location.reload();
                                }
                                if (!json.is_connected && json.error_code) {
                                    clearInterval(checkOauthStatusInterval);
                                    new swal({
                                        type: 'error',
                                        icon: "warning",
                                        dangerMode: true,
                                        title: json.error_code,
                                        text: "<?php esc_html__('Integration has been declined. Please try again', 'woo-xendit-virtual-accounts'); ?>",
                                        buttons: [false, true],
                                        closeOnClickOutside: false,
                                    });
                                }
                            });
                    }, 5000);
                });

                <?php if ($this->developmentmode == 'yes') { ?>
                $('.xendit_dev').parents('tr').show();
                $('.xendit_live').parents('tr').hide();
                <?php } else { ?>
                $('.xendit_dev').parents('tr').hide();
                $('.xendit_live').parents('tr').show();
                <?php } ?>

                <?php if ($this->for_user_id) { ?>
                $("#woocommerce_xendit_gateway_enable_xenplatform").prop('checked', true);
                $('.xendit-xenplatform').parents('tr').show();
                <?php } else { ?>
                $("#woocommerce_xendit_gateway_enable_xenplatform").prop('checked', false);
                $('.xendit-xenplatform').parents('tr').hide();
                <?php } ?>

                $(".xendit-ttl").wrap("<div class='xendit-ttl-wrapper'></div>");
                $("<span class='xendit-form-suffix'>Seconds</span>").insertAfter(".xendit-ttl");

                $(".xendit-ext-id").wrap("<div class='input-text regular-input xendit-ttl-wrapper'></div>");
                $("<span class='xendit-form-suffix'>-order_id</span>").insertAfter(".xendit-ext-id");

                $("#ext-id-example").text(
                    "<?php echo esc_html($this->external_id_format) ?>-4245");

                $("#woocommerce_xendit_gateway_external_id_format").change(
                    function () {
                        $("#ext-id-example").text($(this).val() + "-4245");
                    });

                var isSubmitCheckDone = false;

                $('button[name="save"]').on('click', function (e) {
                    if (isSubmitCheckDone) {
                        isSubmitCheckDone = false;
                        return;
                    }

                    e.preventDefault();

                    //empty "on behalf of" if enable xenplatform is unchecked
                    if (!$("#woocommerce_xendit_gateway_enable_xenplatform").is(":checked")) {
                        $("#woocommerce_xendit_gateway_on_behalf_of").val('');
                    }

                    if ($("#woocommerce_xendit_gateway_external_id_format").length > 0) {
                        var externalIdValue = $("#woocommerce_<?php echo esc_html($this->id); ?>_external_id_format").val();
                        if (externalIdValue.length === 0) {
                            return new swal({
                                type: 'error',
                                title: 'Invalid External ID Format',
                                text: 'External ID cannot be empty, please input one or change it to woo-xendit-virtual-accounts'
                            }).then(function () {
                                e.preventDefault();
                            });
                        }

                        if (/[^a-z0-9-]/gmi.test(externalIdValue)) {
                            return new swal({
                                type: 'error',
                                title: 'Unsupported Character',
                                text: 'The only supported characters in external ID are alphanumeric (a - z, 0 - 9) and dash (-)'
                            }).then(function () {
                                e.preventDefault();
                            });
                        }

                        if (externalIdValue.length <= 5 || externalIdValue.length > 54) {
                            return new swal({
                                type: 'error',
                                title: 'External ID length is outside range',
                                text: 'External ID must be between 6 to 54 characters'
                            }).then(function () {
                                e.preventDefault();
                            });
                        }
                    }

                    isSubmitCheckDone = true;
                    $("button[name='save']").trigger('click');
                });

                $("#woocommerce_xendit_gateway_enable_xenplatform").on('change',
                    function () {
                        if (this.checked) {
                            $(".xendit-xenplatform").parents("tr").show();
                        } else {
                            $(".xendit-xenplatform").parents("tr").hide();
                        }
                    }
                );

                $("#woocommerce_xendit_gateway_developmentmode").on('change',
                    function () {
                        if (this.checked) {
                            $(".xendit_dev").parents("tr").show();
                            $(".xendit_live").parents("tr").hide();
                        } else {
                            $(".xendit_dev").parents("tr").hide();
                            $(".xendit_live").parents("tr").show();
                        }
                    }
                );

                // Overwrite default value
                $("#woocommerce_xendit_gateway_dummy_api_key").val("<?php echo esc_html($this->generateStarChar(strlen($this->get_option('api_key')))); ?>");
                $("#woocommerce_xendit_gateway_dummy_secret_key").val("<?php echo esc_html($this->generateStarChar(strlen($this->get_option('secret_key')))); ?>");
                $("#woocommerce_xendit_gateway_dummy_api_key_dev").val("<?php echo esc_html($this->generateStarChar(strlen($this->get_option('api_key_dev')))); ?>");
                $("#woocommerce_xendit_gateway_dummy_secret_key_dev").val("<?php echo esc_html($this->generateStarChar(strlen($this->get_option('secret_key_dev')))); ?>");
            });
        </script>
        <?php
    }

    /**
     * @return void
     */
    public function init_form_fields()
    {
        $this->form_fields = require(WC_XENDIT_PG_PLUGIN_PATH . '/libs/settings/wc-xendit-gateway-settings.php');
    }

    public function payment_fields()
    {
        if ($this->description) {
            $test_description = '';
            if ($this->developmentmode == 'yes') {
                $test_description = wp_kses(__('<strong>TEST MODE</strong> - Real payment will not be detected', 'woo-xendit-virtual-accounts'), ['strong' => []]);
            }

            echo wp_kses('<p>' . esc_html($this->description) . '</p>
                <p style="color: red; font-size:80%; margin-top:10px;">' . esc_html($test_description) . '</p>', ['p' => ['style' => true]]);
        }
    }

    public function receipt_page($order_id)
    {
        $payment_gateway = wc_get_payment_gateway_by_order($order_id);
        if ($payment_gateway->id != $this->id) {
            return;
        }

        $return = '<div style="text-align:left;"><strong>' . $this->checkout_msg . '</strong><br /><br /></div>';

        $allowed_html = array(
            'div' => array(
                'style' => true
            ),
            'strong' => true,
            'br' => true
        );

        if ($this->developmentmode == 'yes') {
            $testDescription = sprintf(wp_kses(__('<strong>TEST MODE.</strong> The bank account numbers shown below are for testing only. Real payments will not be detected', 'woo-xendit-virtual-accounts'), ['strong' => []]));
            $return .= '<div style="text-align:left;">' . $testDescription . '</div>';
        }

        echo wp_kses($return, $allowed_html);
    }

    public function get_success_redirect_url($order): string
    {
        $returnUrl = $this->get_return_url($order);

        if (strpos($returnUrl, home_url()) === false) {
            $returnUrl = rtrim(wc_get_checkout_url(), '/') . $returnUrl;
        }

        return $returnUrl;
    }

    /**
     * @param $order_id
     * @return array|void
     * @throws Exception
     */
    public function process_payment($order_id)
    {
        try {
            $order = wc_get_order($order_id);
            $amount = $order->get_total();
            $currency = $order->get_currency();

            if ($amount < $this->DEFAULT_MINIMUM_AMOUNT) {
                WC_Xendit_PG_Helper::cancel_order($order, 'Cancelled because amount is below minimum amount');
                /* translators: %1%s: Currency, 2: Min Amount. */
                $err_msg = sprintf(__(
                    'The minimum amount for using this payment is %1$s %2$s. Please put more item(s) to reach the minimum amount. Code: 100001',
                    'woo-xendit-virtual-accounts'
                ), $currency, wc_price($this->DEFAULT_MINIMUM_AMOUNT));

                wc_add_notice($this->get_localized_error_message('INVALID_AMOUNT_ERROR', $err_msg), 'error');

                return array(
                    'result' => 'failure',
                    'message' => $err_msg,
                );
            }

            if ($amount > $this->DEFAULT_MAXIMUM_AMOUNT) {
                WC_Xendit_PG_Helper::cancel_order($order, 'Cancelled because amount is above maximum amount');

                /* translators: %1%s: Currency, 2: Max Amount. */
                $err_msg = sprintf(__(
                    'The maximum amount for using this payment is %1$s %2$s. Please remove one or more item(s) from your cart. Code: 100002',
                    'woo-xendit-virtual-accounts'
                ), $currency, wc_price($this->DEFAULT_MAXIMUM_AMOUNT));

                wc_add_notice($this->get_localized_error_message('INVALID_AMOUNT_ERROR', $err_msg), 'error');

                return array(
                    'result' => 'failure',
                    'message' => $err_msg,
                );;
            }

            $blog_name = html_entity_decode(get_option('blogname'), ENT_QUOTES | ENT_HTML5);
            $description = WC_Xendit_PG_Helper::generate_invoice_description($order);

            $payer_email = !empty($order->get_billing_email()) ? $order->get_billing_email() : 'noreply@mail.com';
            $payment_gateway = wc_get_payment_gateway_by_order($order_id);

            // How likely this condition below will happened?
            if ($payment_gateway->id != $this->id) {
                return array(
                    'result' => 'failure',
                    'message' => 'Can\'t proceed the order with '.$payment_gateway->id,
                );
            }

            $invoice = $order->get_meta('Xendit_invoice');
            $invoice_exp = $order->get_meta('Xendit_expiry');

            $additional_data = WC_Xendit_PG_Helper::generate_items_and_customer($order);
            $invoice_data = array(
                'external_id' => WC_Xendit_PG_Helper::generate_external_id($order, $this->external_id_format),
                'amount' => $amount,
                'currency' => $currency,
                'payer_email' => $payer_email,
                'description' => $description,
                'client_type' => 'INTEGRATION',
                'success_redirect_url' => $this->get_success_redirect_url($order),
                'failure_redirect_url' => wc_get_checkout_url(),
                'platform_callback_url' => $this->xendit_callback_url,
                'checkout_redirect_flow' => $this->redirect_after,
                'customer' => !empty($additional_data['customer']) ? $additional_data['customer'] : '',
                'items' => !empty($additional_data['items']) ? $additional_data['items'] : ''
            );

            // Generate Xendit payment fees
            $fees = WC_Xendit_Payment_Fees::generatePaymentFees($order);
            if (!empty($fees)) {
                $invoice_data['fees'] = $fees;
            }

            $header = array(
                'x-plugin-method' => strtoupper($this->method_code),
                'x-plugin-store-name' => $blog_name
            );

            if ($invoice && $invoice_exp > time()) {
                $response = $this->xenditClass->getInvoice($invoice);
            } else {
                $response = $this->xenditClass->createInvoice($invoice_data, $header);
            }

            if (!empty($response['error_code'])) {
                $response['message'] = !empty($response['code']) ? $response['message'] . ' Code: ' . $response['code'] : $response['message'];
                $message = $this->get_localized_error_message($response['error_code'], $response['message']);
                $order->add_order_note('Checkout with invoice unsuccessful. Reason: ' . $message);

                throw new Exception($message);
            }

            if ($response['status'] == 'PAID' || $response['status'] == 'COMPLETED') {
                // Return thankyou redirect
                return array(
                    'result'    => 'success',
                    'redirect'  => $this->get_return_url($order)
                );
            }

            $xendit_invoice_url = esc_attr($response['invoice_url']);
            $order->update_meta_data('Xendit_invoice', esc_attr($response['id']));
            $order->update_meta_data('Xendit_invoice_url', $xendit_invoice_url);
            $order->update_meta_data('Xendit_expiry', esc_attr(strtotime($response['expiry_date'])));
            $order->save();

            switch ($this->redirect_after) {
                case 'ORDER_RECEIVED_PAGE':
                    $args = array(
                        'utm_nooverride' => '1',
                        'order_id' => $order_id,
                    );
                    $return_url = esc_url_raw(add_query_arg($args, $this->get_return_url($order)));
                    break;
                case 'CHECKOUT_PAGE':
                default:
                    $return_url = $xendit_invoice_url;
            }

            // clear cart session
            if (WC()->cart) {
                WC()->cart->empty_cart();
            }

            // Return thankyou redirect
            return array(
                'result' => 'success',
                'redirect' => $return_url,
            );
        } catch (Throwable $e) {
            if ($e instanceof Exception) {
                wc_add_notice($e->getMessage(), 'error');
            }
            $metrics = $this->xenditClass->constructMetricPayload('woocommerce_checkout', array(
                'type' => 'error',
                'payment_method' => strtoupper($this->method_code),
                'error_message' => $e->getMessage()
            ));
            $this->xenditClass->trackMetricCount($metrics);

            return array(
                'result' => 'failure',
                'message' => $e->getMessage(),
            );
        }
    }

    /**
     * @param $response
     * @return void
     */
    public function validate_payment($response)
    {
        global $wpdb, $woocommerce;

        try {
            $external_id = $response->external_id;
            $exploded_ext_id = explode("-", $external_id);
            $order_num = end($exploded_ext_id);

            if (!is_numeric($order_num)) {
                $exploded_ext_id = explode("_", $external_id);
                $order_num = end($exploded_ext_id);
            }

            if (WC_Xendit_PG_Helper::is_advanced_order_number_active()) {
                // 1. Try direct meta query
                $orders = wc_get_orders(array(
                    'meta_key' => '_order_number',
                    'meta_value' => $order_num,
                    'limit' => 1
                ));
                
                if (!empty($orders) && count($orders) == 1) {
                    $order = $orders[0];
                    $order_num = $order->get_id();
                }
            }

            $order = wc_get_order($order_num);
            $order_id = $order->get_id();

            if ($this->developmentmode != 'yes') {
                $payment_gateway = wc_get_payment_gateway_by_order($order_id);
                if (false === get_post_status($order_id) || strpos($payment_gateway->id, 'xendit')) {
                    header('HTTP/1.1 400 Invalid Data Received');
                    die('Xendit is live and require a valid order id');
                }
            }

            if (in_array($response->status, array('PAID', 'SETTLED'))) {
                //update payment method in case customer change method after invoice is generated
                $order->set_payment_method($this->id);
                $order->set_payment_method_title($this->method_title);

                //save charge ID if paid by credit card
                if ($response->channel == 'CREDIT_CARD' && !empty($response->credit_card_charge_id)) {
                    $order->set_transaction_id($response->credit_card_charge_id);
                }

                $order->save();

                $notes = WC_Xendit_PG_Helper::build_order_notes(
                    $response->id,
                    $response->status,
                    $response->channel,
                    $order->get_currency(),
                    $order->get_total()
                );
                WC_Xendit_PG_Helper::complete_payment($order, $notes, $this->success_payment_xendit);

                // Empty cart in action
                $woocommerce->cart->empty_cart();

                die('Success');
            } else {
                if (empty($order->get_meta('Xendit_invoice_expired'))) {
                    $order->add_meta_data('Xendit_invoice_expired', 1);
                }
                $order->update_status('failed');

                $notes = WC_Xendit_PG_Helper::build_order_notes(
                    $response->id,
                    $response->status,
                    $response->channel,
                    $order->get_currency(),
                    $order->get_total()
                );

                $order->add_order_note( wp_kses("<b>Xendit payment failed.</b><br>", ['b' => true, 'br' => true]). $notes);
                die(esc_html('Invoice ' . $response->channel . ' status is ' . $response->status));
            }
        } catch (Exception $e) {
            header('HTTP/1.1 500 Server Error');
            echo esc_html($e->getMessage());
            exit;
        }
    }

    public function check_gateway_status($gateways)
    {
        global $woocommerce;

        if (is_null($woocommerce->cart)) {
            return $gateways;
        }

        if ($this->enabled == 'no') {
            // Disable all Xendit payments
            if ($this->id == 'xendit_gateway') {
                return array_filter($gateways, function ($gateway) {
                    return strpos($gateway->id, 'xendit') === false;
                });
            }

            unset($gateways[$this->id]);
            return $gateways;
        }

        if (!$this->xenditClass->isCredentialExist()) {
            unset($gateways[$this->id]);
            return $gateways;
        }

        /**
         * get_cart_contents_total() will give us just the final (float) amount after discounts.
         * Compatible with WC version 3.2.0 & above.
         * Source: https://woocommerce.github.io/code-reference/classes/WC-Cart.html#method_get_cart_contents_total
         */
        $amount = $woocommerce->cart->get_total('');
        if ($amount > $this->DEFAULT_MAXIMUM_AMOUNT) {
            unset($gateways[$this->id]);
            return $gateways;
        }

        if (WC_Xendit_PG_Helper::is_cart_contain_subscription_items()) {
            unset($gateways[$this->id]);
            return $gateways;
        }

        return $gateways;
    }

    /**
     * Return filter of PG icon image in checkout page. Called by this class automatically.
     */
    public function get_icon()
    {
        $style = "style='margin-left: 0.3em; max-height: 28px; max-width: 65px;'";
        $icon = '<img src="' . plugins_url('assets/images/xendit.svg', WC_XENDIT_PG_MAIN_FILE) . '" alt="Xendit" ' . $style . ' />';

        return apply_filters('woocommerce_gateway_icon', $icon, $this->id);
    }

    /**
     * @return string
     */
    public function get_xendit_admin_description(): string
    {
        return $this->method_description;
    }

    public function get_xendit_title() {
        return !empty($this->get_option('channel_name')) ? $this->get_option('channel_name') : $this->default_title;
    }

    public function get_xendit_description() {
        return !empty($this->get_option('payment_description')) ? nl2br($this->get_option('payment_description')) : esc_html(__('Pay your order via Xendit Payment Gateway', 'woo-xendit-virtual-accounts'));
    }

    /**
     * @param $sub_account_id
     * @return true
     * @throws Exception
     */
    protected function validate_sub_account($sub_account_id): bool
    {
        if (empty($sub_account_id)) {
            throw new Exception(esc_html('Please enter XenPlatform User.'));
        }

        $response = $this->xenditClass->getSubAccount($sub_account_id);
        if (!empty($response['account_id'])) {
            return true;
        }

        if (!empty($response['error_code'])) {
            throw new Exception(esc_html($response['message']));
        }

        throw new Exception(esc_html('Validate XenPlatform User failed'));
    }

    /**
     * @param array $settings
     * @return bool
     */
    protected function is_test_mode(array $settings = []): bool
    {
        if (empty($settings['secret_key']) && !empty($settings['secret_key_dev'])) {
            return true;
        }

        return false;
    }

    /**
     * @return bool
     * @throws Exception
     */
    public function process_admin_options(): bool
    {
        // To avoid duplicated request
        if (self::$setting_processed > 0) {
            return false;
        }

        $this->init_settings();
        $post_data = $this->get_post_data();

        foreach ($this->get_form_fields() as $key => $field) {
            if ('title' !== $this->get_field_type($field)) {
                try {
                    $value = $this->get_field_value($key, $field, $post_data);

                    // map dummy api keys
                    if (in_array($key, self::API_KEY_FIELDS)) {
                        $real_key_field = str_replace('dummy_', '', $key);
                        $real_api_key_char_count = !empty($this->settings[$real_key_field]) ? strlen($this->settings[$real_key_field]) : 0;

                        if ($value === $this->generateStarChar($real_api_key_char_count)) { // skip when no changes
                            continue;
                        } else {
                            $this->settings[$real_key_field] = $value; // save real api keys in original field name
                        }
                        $this->settings[$key] = $this->generateStarChar($real_api_key_char_count); // always set dummy fields to ****
                        continue;
                    }

                    $this->settings[$key] = $value;
                } catch (Exception $e) {
                    WC_Admin_Settings::add_error(esc_html($e->getMessage()));
                }
            }
        }

        if (!isset($post_data['woocommerce_' . $this->id . '_enabled']) && $this->get_option_key() == 'woocommerce_' . $this->id . '_settings') {
            $this->settings['enabled'] = $this->id === 'xendit_gateway' ? 'no' : $this->enabled;
        }

        // default value
        if ($this->id === 'xendit_gateway') {
            $this->settings['external_id_format'] = empty($this->settings['external_id_format']) ? self::DEFAULT_EXTERNAL_ID_VALUE : $this->settings['external_id_format'];
        }

        // Update settings
        update_option($this->get_option_key(), apply_filters('woocommerce_settings_api_sanitized_fields_' . $this->id, $this->settings), 'yes');
        self::$setting_processed += 1;

        // validate sub account
        try {
            if (isset($this->settings['enable_xenplatform']) && $this->settings['enable_xenplatform'] === 'yes') {
                $this->validate_sub_account($this->settings['on_behalf_of']);
            }
        } catch (Exception $e) {
            // Reset Xen Platform if validation failed
            $this->settings['enable_xenplatform'] = 'no';
            $this->settings['on_behalf_of'] = '';
            update_option($this->get_option_key(), apply_filters('woocommerce_settings_api_sanitized_fields_' . $this->id, $this->settings), 'yes');

            WC_Admin_Settings::add_error(esc_html($e->getMessage()));
            return false;
        }

        return true;
    }

    /**
     * @param $count
     * @return string
     */
    private function generateStarChar($count = 0): string
    {
        $result = '';
        for ($i = 0; $i < $count; $i++) {
            $result .= '*';
        }

        return $result;
    }

    public function get_localized_error_message($error_code, $message)
    {
        switch ($error_code) {
            case 'UNSUPPORTED_CURRENCY':
                return str_replace('{{currency}}', get_woocommerce_currency(), $message);
            default:
                return $message ? $message : $error_code;
        }
    }

    /**
     * @return string
     */
    public function get_xendit_option(string $key)
    {
        return $this->get_option($key);
    }

    /**
     * @param $order_id
     * @param $old_status
     * @param $new_status
     * @return void
     * @throws Exception
     */
    public function expire_invoice_when_order_cancelled($order_id, $old_status, $new_status)
    {
        if ($new_status !== 'cancelled') {
            return;
        }

        $order = wc_get_order($order_id);
        if ($order) {
            $payment_method = $order->get_payment_method();
            $xendit_invoice_expired = $order->get_meta('Xendit_invoice_expired');
            $xendit_invoice_id = $order->get_meta('Xendit_invoice');

            if (preg_match('/xendit/i', $payment_method)
                && empty($xendit_invoice_expired)
            ) {
                // Expire Xendit invoice
                $response = $this->xenditClass->expiredInvoice($xendit_invoice_id);
                if (!empty($response) && !isset($response['error_code'])) {
                    $order->add_meta_data('Xendit_invoice_expired', 1);
                    $order->save();
                }
            }
        }
    }

    /**
     * Cancel all unpaid orders after held duration to prevent stock lock for those products.
     */
    public function custome_cancel_unpaid_orders()
    {
        global $wpdb;

        $held_duration = get_option('woocommerce_hold_stock_minutes');

        if ($held_duration < 1 || 'yes' !== get_option('woocommerce_manage_stock')) {
            return;
        }

        $canceled_order = $wpdb->get_col(
            $wpdb->prepare(
				// @codingStandardsIgnoreStart
				"SELECT posts.ID
				FROM {$wpdb->posts} AS posts
                LEFT JOIN {$wpdb->postmeta} AS pm_expired
                    ON posts.ID = pm_expired.post_id
                        AND pm_expired.meta_key = 'Xendit_invoice_expired'
                LEFT JOIN {$wpdb->postmeta} AS pm_method
                    ON posts.ID = pm_method.post_id
                        AND pm_method.meta_key = '_payment_method'
				WHERE posts.post_type IN ('" . implode( "','", wc_get_order_types() ) . "')
                    AND posts.post_status = 'wc-cancelled'
                    AND `pm_method`.`meta_value` LIKE 'xendit_%'
                    AND pm_expired.meta_id IS NULL"
			)
		);

        if ($canceled_order) {
            foreach ($canceled_order as $cancel_order) {
                $order = wc_get_order( $cancel_order );
                $xendit_invoice_expired = $order->get_meta('Xendit_invoice_expired');
                $xendit_invoice_id = $order->get_meta('Xendit_invoice');
                if (empty($xendit_invoice_expired)) {
                    $order->add_meta_data('Xendit_invoice_expired', 1);
                    $order->save();

                    $response = $this->xenditClass->expiredInvoice($xendit_invoice_id);
                }
            }
        }
    }

    /**
     * @param $public_key
     * @param $public_key_dev
     * @return true
     */
    public function update_public_keys($public_key, $public_key_dev): bool
    {
        if (!empty($public_key)) {
            $this->update_option('api_key', $public_key);
        }

        if (!empty($public_key_dev)) {
            $this->update_option('api_key_dev', $public_key_dev);
        }

        return true;
    }
}
